/**
 * \file keystore.h
 *
 * \brief Functions and defines to manage the key-store
 *
 * \author Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

#ifndef __SDC_KEYSTORE_H_
#define __SDC_KEYSTORE_H_

#include <sdc.h>
#include <keyman.h>
#include "daemon.h"

#define NEW_KEYSTORE_MODE 0600

/** File path variations */
typedef enum {
    KEYSTORE_STANDARD,  /**< Base path */
    KEYSTORE_PENDING,   /**< Base path + ".pending" */
    KEYSTORE_END
} keystore_name_variation_t;


/** Keystore ctx containing configuration dir handle and keystore names */
typedef struct keystore_ctx_t {
    /** key_store paths */
    char *keystore_names[KEYSTORE_END];
    int keystore_dir_fd;

    libkeyman_ctx_t *libkeyman_ctx;
} keystore_ctx_t;

typedef struct automatic_kid_range_t {
    sdc_key_id_t first;
    sdc_key_id_t last;
} automatic_kid_range_t;

/**
 * Initialize the keystore context
 *
 * \param ctx       Keystore context
 * \param keystore_path  Path to the keystore basename
 * \return corresponding sdc_error
 */
sdc_error_t keystore_ctx_init(keystore_ctx_t *ctx, const char *keystore_path);

/**
 * Free the keystore context
 *
 * \param ctx       Keystore context
 * \return corresponding sdc_error
 */
sdc_error_t keystore_ctx_fini(keystore_ctx_t *ctx);

/**
 * Initialize the keystore
 *
 * In case no keystore exists this function will create an empty keystore
 * This function must only be called after the ctx has
 * been initialized using \ref keystore_ctx_init
 *
 * \param ctx       Keystore context
 * \return SDC_KEY_STORAGE_ALREADY_PRESENT in case a normal or pending keystore exists
 * \return otherwise corresponding sdc_error
 */
sdc_error_t keystore_initialize(keystore_ctx_t *ctx);

/**
 * Verify that the SDC daemon has the required permissions for keystores and
 * its folders.
 *
 * A warning is generated in case write permissions are missing (e.g. read only fs)
 *
 * This function must only be called after the ctx has
 * been initialized using \ref keystore_ctx_init
 *
 * \param ctx       Keystore context
 * \return corresponding sdc_error
 */
sdc_error_t keystore_verify_credentials(keystore_ctx_t *ctx);

/**
 * Restore the content of the keystore and load the keys into kernel
 *
 * In case only the pending keystore exists the pending keystore is loaded
 * and afterwards renamed to normal keystore.
 * (system rebooted before pending was renamed to normal keystore - but as
 * normal keystore is already deleted, the pending keystore is complete)
 *
 * Otherwise the normal keystore is loaded and pending keystore is discarded
 * ( in case both exist it is unclear if the pending keystore has been updated
 * completely - therefore it is discarded )
 *
 * After successful call of this function only a normal keystore will exist
 *
 * This function must only be called after the ctx has
 * been initialized using \ref keystore_ctx_init
 *
 * \param ctx       Keystore context
 * \return corresponding sdc_error
 */
sdc_error_t keystore_restore_persistent(keystore_ctx_t *ctx);

/**
 * Unlock the keystore so that its keys can be referenced
 *
 * This function must only be called after the ctx has
 * been initialized using \ref keystore_ctx_init
 *
 * \param ctx       Keystore context
 * \return corresponding sdc_error
 */
sdc_error_t keystore_unlock(keystore_ctx_t *ctx);

/**
 *  Overwrite the buffer containing secret data with 0
 *
 *  The asm volatile with memory in the clobber list is intended
 *  to ensure that the compiler does not reorder instructions
 *  and removes memset in optimization.
 *
 * \param secret    Buffer containing potentially secret data
 * \param secretlen Length of the buffer
 */
void keystore_overwrite_secret(uint8_t *secret, size_t secretlen);

/**
 * Install a keystore key
 *
 * In case the key is a persistent key the keystore will be updated too
 * Otherwise the keystore is not modified
 *
 * This function must only be called after the ctx has
 * been initialized using \ref keystore_ctx_init
 *
 * \param[in] ctx       Keystore context
 * \param[in] kid_min   min desired key ID
 * \param[in] kid_max   max desired key ID - set \p kid_min == \p kid_max to request explicit key ID
 * \param[out] out_kid	Pointer to return got key ID
 * \param[in] key_spec  struct containing format, length and permission settings
 * \param[in] persistent Install the key persistently (saved to keystore) or just volatile (available till next reboot)
 * \param[in] key_data_enc externally defined encoding of \p key_data
 * \param[in] key_data  Buffer containing data of the key if == NULL a random key else normal key
 * \param[in] key_data_len length of key_data buffer
 *
 * \return SDC_KEY_INVALID - in case \p kid_min or \p kid_max are invalid or \p kid_min > kid_max
 * \return corresponding sdc_error
 */
sdc_error_t keystore_install_plain_or_random_key(keystore_ctx_t *ctx,
                                                 sdc_key_id_t kid_min,
                                                 sdc_key_id_t kid_max,
                                                 sdc_key_id_t *out_kid,
                                                 const libkeyman_key_spec_t *key_spec,
                                                 bool persistent,
                                                 sdc_keysecret_enc_t key_data_enc,
                                                 const uint8_t *key_data,
                                                 size_t key_data_len);

/**
 * Import a wrapped key into the keystore
 *
 * Import a wrapped key into the keystore consists out
 * of two steps.
 * 1st: sdc_arch_wrapped_storage_key_to_daemon_data is
 * used to generate architecture specific daemon_data.
 * 2nd(this function): Installing the key from architecture
 * specific daemon data(key_data) to the keystore.
 *
 * The other parameters are simply forwarded.
 *
 * In case the key is a persistent key the keystore will be updated too
 * Otherwise the keystore is not modified
 *
 * This function must only be called after the ctx has
 * been initialized using \ref keystore_ctx_init
 *
 * \param[in] ctx       Keystore context
 * \param[in] kid_min   min desired key ID
 * \param[in] kid_max   max desired key ID - set \p kid_min == \p kid_max to request explicit key ID
 * \param[out] out_kid  Pointer to return got key ID
 * \param[in] key_spec  struct containing format, length and permission settings
 * \param[in] persistent Install the key persistently (saved to keystore) or just volatile (available till next reboot)
 * \param[in] key_data_enc externally defined encoding of \p key_data
 * \param[in] key_data  Buffer containing data of the key if == NULL a random key else normal key
 * \param[in] key_data_len length of key_data buffer
 *
 * \return SDC_KEY_INVALID - in case \p kid_min or \p kid_max are invalid or \p kid_min > kid_max
 * \return corresponding sdc_error
 */
sdc_error_t keystore_import_key(keystore_ctx_t *ctx,
                                sdc_key_id_t kid_min,
                                sdc_key_id_t kid_max,
                                sdc_key_id_t *out_kid,
                                const libkeyman_key_spec_t *key_spec,
                                bool persistent,
                                sdc_keysecret_enc_t key_data_enc,
                                const uint8_t *key_data,
                                size_t key_data_len);

/**
 * Install a insecure product key
 * These keys are always persistent and have explicitly specified key id.
 * Furthermore the key secret is always specified explicitly
 *
 * This function must only be called after the ctx has
 * been initialized using \ref keystore_ctx_init
 *
 * \param ctx       Keystore context
 * \param key_secret        key secret of the product key
 * \param key_secret_len    length of \p key_secret
 * \param cred		Credentials of the process calling install
 * \param kid		Pointer to key ID of key to install
 *                  In case \p automatic_id = false - if is read from kid
 *                  In case \p automatic_id = true - random ID is determined internally and returned using kid
 * \return corresponding sdc_error
 */
sdc_error_t keystore_install_product_key(keystore_ctx_t *ctx,
                                         uint8_t *key_secret, size_t key_secret_len,
                                         daemon_credentials_t *cred,
                                         sdc_key_id_t kid);

/**
 * Remove a keystore key
 *
 * In case the key is a persistent key the keystore will be updated too
 * Otherwise the keystore is not modified
 *
 * This function must only be called after the ctx has
 * been initialized using \ref keystore_ctx_init
 *
 * \param ctx       Keystore context
 * \param kid		Id of the key to be removed
 * \param cred		Credentials of the process calling remove
 *
 * \return corresponding sdc_error
 */
sdc_error_t keystore_remove_key(keystore_ctx_t *ctx, sdc_key_id_t kid,
                                daemon_credentials_t *cred);
#endif
